/*
 * Copyright European Commission's
 * Taxation and Customs Union Directorate-General (DG TAXUD).
 */
package eu.europa.ec.taxud.cesop.readers;

import javax.xml.namespace.QName;
import javax.xml.stream.XMLStreamException;

import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

import org.codehaus.stax2.validation.XMLValidationSchema;

import eu.europa.ec.taxud.cesop.utils.ValidationConstants.XML;

/**
 * Utils class reading an XML file.
 */
public class CesopXmlReader implements AutoCloseable {

    private final XmlStreamReaderWrapper xmlStreamReaderWrapper;
    private final Map<String, String> values = new HashMap<>();

    public CesopXmlReader(InputStream inputStream) throws XMLStreamException {
        this(inputStream, null);
    }

    /**
     * Constructor.
     *
     * @param inputStream the input stream. The input stream is not closed by {@link CesopXmlReader}.
     * @param xsd         the xsd
     * @throws XMLStreamException in case of error while processing the XML content
     */
    public CesopXmlReader(InputStream inputStream, XMLValidationSchema xsd) throws XMLStreamException {
        this.xmlStreamReaderWrapper = new XmlStreamReaderWrapper(inputStream, xsd);
    }

    /**
     * Returns the current start tag name.
     *
     * @return the tag name
     * @throws XMLStreamException if the current event type is not a start element
     */
    public QName getStartElementName() throws XMLStreamException {
        return this.xmlStreamReaderWrapper.getStartElementName();
    }

    /**
     * Positions the cursor on the next XML start element..
     * The next call to {@link XmlStreamReaderWrapper#moveToNextElement()} will return the start element.
     *
     * @return true if the start element is found, false otherwise
     * @throws XMLStreamException in case of error while processing the XML content
     */
    public boolean positionCursorOnStartElement() throws XMLStreamException {
        final boolean result = this.xmlStreamReaderWrapper.goToNextStartElement();
        this.xmlStreamReaderWrapper.markAsPeek();
        return result;
    }

    /**
     * Positions the cursor on the next XML start element with specified QName.
     * The next call to {@link XmlStreamReaderWrapper#moveToNextElement()} will return the start element.
     *
     * @param qName the name of the start element to search for
     * @return true if the start element is found, false otherwise
     * @throws XMLStreamException in case of error while processing the XML content
     */
    public boolean positionCursorOnStartElement(final QName qName) throws XMLStreamException {
        final boolean result = this.xmlStreamReaderWrapper.goToNextStartElement(qName);
        this.xmlStreamReaderWrapper.markAsPeek();
        return result;
    }

    /**
     * If the next start element is qName, then returns an {@link Optional} containing {@link Map} with all the children values.
     * Otherwise, returns an empty {@link Optional}.
     *
     * @param qName the tag
     * @return the map of values
     * @throws XMLStreamException in case of error while processing the XML content
     */
    public Optional<Map<String, String>> readNextTagIfEquals(final QName qName) throws XMLStreamException {
        this.values.clear();
        if (this.positionCursorOnStartElement() && this.xmlStreamReaderWrapper.getStartElementName().equals(qName)) {
            this.readTagElements(qName, XML.TAG_NAME_SEPARATOR + qName.getLocalPart());
            return Optional.of(this.values);
        }
        return Optional.empty();
    }

    /**
     * Read the next tag QName and return a map with all the children values.
     *
     * @param qName the tag
     * @return the map of values
     * @throws XMLStreamException in case of error while processing the XML content
     */
    public Map<String, String> readNextTagIntoMap(final QName qName) throws XMLStreamException {
        this.values.clear();
        if (this.positionCursorOnStartElement(qName)) {
            this.readTagElements(qName, XML.TAG_NAME_SEPARATOR + qName.getLocalPart());
            return this.values;
        }
        throw new XMLStreamException("Start tag not found: " + qName);
    }

    private void readTagElements(final QName qName, final String entryKey) throws XMLStreamException {
        while (this.xmlStreamReaderWrapper.hasNext()) {
            this.xmlStreamReaderWrapper.moveToNextElement();
            if (this.xmlStreamReaderWrapper.isEndElement() && this.xmlStreamReaderWrapper.getEndElementName().equals(qName)) {
                return;
            } else if (this.xmlStreamReaderWrapper.isStartElement()) {
                final QName elementName = this.xmlStreamReaderWrapper.getStartElementName();
                final String key = entryKey + XML.TAG_NAME_SEPARATOR + elementName.getLocalPart();
                if (!elementName.equals(qName)) {
                    this.readAttributes(key);
                    this.readTagElements(elementName, key);
                } else {
                    this.readAttributes(entryKey);
                }
            } else if (this.xmlStreamReaderWrapper.isCharactersElement()) {
                final String data = this.xmlStreamReaderWrapper.getCharacters().trim();
                if (data.length() > 0) {
                    this.values.put(entryKey, data);
                }
            }
        }
        throw new XMLStreamException("End tag not found: " + qName);
    }

    private void readAttributes(final String tagName) throws XMLStreamException {
        this.xmlStreamReaderWrapper.getAttributes().forEach(
                (key, value) -> this.values.put(tagName + XML.ATTRIBUTE_NAME_SEPARATOR + key, value));
    }

    /**
     * Returns the XML stream reader wrapper.
     *
     * @return the XML stream reader wrapper
     */
    public XmlStreamReaderWrapper getXmlStreamReaderWrapper() {
        return this.xmlStreamReaderWrapper;
    }

    @Override
    public void close() throws XMLStreamException {
        this.xmlStreamReaderWrapper.close();
    }
}
